let tableBody = document.querySelector("#table tbody");
let selectAllBtn = document.querySelector("#selectAllBtn");
let addBtn = document.querySelector("#addBtn");
let saveBtn = document.querySelector("#saveBtn");
let deleteBtn = document.querySelector("#deleteBtn");
let printBtn = document.querySelector("#printBtn");

// Load records into the table
async function loadRecords() {
  try {
    console.log("loadRecords function started");

    // Fetch records from the server
    let response = await fetch("load_records.php");
    let records = await response.json();

    // Clear the table body
    tableBody.innerHTML = "";

    // Loop through the records and create rows
    records.forEach((record) => {
      let tr = document.createElement("tr");
      tr.dataset.id = record.id; // Store record ID in dataset

      // Construct the row HTML
      tr.innerHTML = `
        <td><input type="checkbox" class="select-checkbox"></td>
        <td>${record.id}</td>
        <td contenteditable="true" data-field="tag_number">${record.tag_number}</td>
        <td contenteditable="true" data-field="category_name">${record.category_name || ""}</td>
        <td contenteditable="true" data-field="title">${record.title || ""}</td>
        <td contenteditable="true" data-field="subtitle">${record.subtitle || ""}</td>
        <td contenteditable="true" data-field="category_id">${record.category_id || ""}</td>
        <td contenteditable="true" data-field="area_id">${record.area_id || ""}</td>
        <td contenteditable="true" data-field="labelSetId">${record.labelSetId || ""}</td>
        <td contenteditable="false" data-field="container_contents">
          ${
            record.container_contents
              ? `<span class="edit-icon" title="${record.container_contents}">📝</span>` // Edit icon with tooltip
              : `<span class="add-icon">➕</span>` // "+" icon
          }
        </td>
      `;

      // Append the row to the table body
      tableBody.appendChild(tr);
    });

    // Attach event listeners to editable cells
    attachInputListeners();

    console.log("Final table body innerHTML:", tableBody.innerHTML);
  } catch (error) {
    console.error("Error in loadRecords function:", error);
  }
}

// Attach event listeners to all editable cells
function attachInputListeners() {
  let tds = tableBody.querySelectorAll("td[contenteditable='true']");
  tds.forEach((td) => {
    td.addEventListener("input", (e) => {
      e.target.classList.add("changed"); // Mark the cell as changed
      enableSaveBtn(); // Enable the SAVE button
    });
  });
}

// Select or de-select all records
function selectAll() {
  let checkboxes = document.querySelectorAll(".select-checkbox");
  let selectAllLabel = selectAllBtn.textContent;
  if (selectAllLabel === "SELECT ALL") {
    checkboxes.forEach((checkbox) => (checkbox.checked = true));
    selectAllBtn.textContent = "DE-SELECT ALL";
  } else {
    checkboxes.forEach((checkbox) => (checkbox.checked = false));
    selectAllBtn.textContent = "SELECT ALL";
  }
  enableDeletePrintBtn();
}

// Add a new blank row
function addRow() {
  let tr = document.createElement("tr");
  tr.dataset.new = "true"; // Add data-new attribute
  tr.innerHTML = `
    <td><input type="checkbox" class="select-checkbox"></td>
    <td></td>
    <td contenteditable="true" data-field="tag_number"></td>
    <td contenteditable="true" data-field="category_name"></td>
    <td contenteditable="true" data-field="title"></td>
    <td contenteditable="true" data-field="subtitle"></td>
    <td contenteditable="true" data-field="category_id"></td>
    <td contenteditable="true" data-field="area_id"></td>
    <td contenteditable="true" data-field="labelSetId"></td>
    <td>Z</td>
  `;
  tableBody.appendChild(tr);

  // Attach input event listener to each editable td in the new row
  attachInputListeners();

  tr.querySelector("td:nth-child(3)").focus();
  enableSaveBtn();
}

// Enable/disable the "SAVE", "DELETE", and "PRINT" buttons
function enableDeletePrintBtn() {
  let checkedCheckboxes = document.querySelectorAll(".select-checkbox:checked");
  deleteBtn.disabled = checkedCheckboxes.length === 0;
  printBtn.disabled = checkedCheckboxes.length === 0;
}

function enableSaveBtn() {
  saveBtn.disabled = false;
}

function disableSaveBtn() {
  saveBtn.disabled = true;
}

// Get all changes to be saved
function getChanges() {
  let changes = [];
  let rows = tableBody.querySelectorAll("tr");

  rows.forEach((row) => {
    if (row.dataset.new === "true") {
      let record = {};
      let fields = row.querySelectorAll("td[contenteditable='true']");
      fields.forEach((field) => {
        if (field.dataset.field) {
          record[field.dataset.field] = field.textContent;
        }
      });
      changes.push({ newRecord: true, record: record });
    } else {
      let fields = row.querySelectorAll("td.changed");
      fields.forEach((field) => {
        changes.push({
          id: row.dataset.id,
          field: field.dataset.field,
          newValue: field.textContent,
          oldValue: field.dataset.oldValue,
        });
      });
    }
  });

  return changes;
}

// Save changes to the database
function saveChanges() {
  let changes = getChanges();

  if (changes.length > 0) {
    let data = new FormData();
    data.append("changes", JSON.stringify(changes));

    fetch("save_changes.php", {
      method: "POST",
      body: data,
    })
      .then((response) => response.json())
      .then((response) => {
        if (response.success) {
          location.reload();
        } else {
          console.error(response.error);
        }
      });
  }
}

// Delete selected rows
async function deleteRows() {
  try {
    if (!confirm("Are you sure you want to delete the selected rows?")) {
      return;
    }

    let checkedCheckboxes = document.querySelectorAll(".select-checkbox:checked");
    if (checkedCheckboxes.length === 0) {
      alert("Please select rows to delete");
      return;
    }

    let idsToDelete = Array.from(checkedCheckboxes).map((checkbox) => {
      return checkbox.closest("tr").dataset.id;
    });

    const response = await fetch("delete_rows.php", {
      method: "POST",
      body: JSON.stringify(idsToDelete),
      headers: { "Content-Type": "application/json" },
    });

    if (response.ok) {
      await loadRecords(); // Reload the table
      disableSaveBtn();
      selectAllBtn.textContent = "SELECT ALL";
      enableDeletePrintBtn(); // Reset button states
    } else {
      throw new Error("Delete operation failed");
    }
  } catch (error) {
    console.error("Error in deleteRows:", error);
    alert("Failed to delete rows: " + error.message);
  }
}

// Print selected rows
function printRows() {
  let checkedCheckboxes = document.querySelectorAll(".select-checkbox:checked");
  if (checkedCheckboxes.length === 0) {
    alert("Please select rows to print");
    return;
  }

  let idsToPrint = Array.from(checkedCheckboxes).map((checkbox) => {
    return checkbox.closest("tr").dataset.id;
  });

  let url = `displaySelectedLabels.php?ids=${idsToPrint.join(",")}`;
  window.open(url, "_blank");
}

// Event listeners
document.addEventListener("DOMContentLoaded", () => {
  loadRecords();

  tableBody.addEventListener("change", enableDeletePrintBtn);
  selectAllBtn.addEventListener("click", selectAll);
  addBtn.addEventListener("click", addRow);
  saveBtn.addEventListener("click", saveChanges);
  deleteBtn.addEventListener("click", deleteRows);
  printBtn.addEventListener("click", printRows);
});
